// $Id: CClassWriterWindow.cpp,v 1.5 2007/02/11 02:20:05 paul Exp $

/*
 * All contents of this source code are copyright 2007 Exp Digital Uk.
 * This file forms part of the example projects for the Infinity API.
 * You may not redistribute either the source code or the compiled binary.
 * The Infinity API is covered by a fair use licence which you should have recieved with this
 * file. If you didnt receieve the licence, please contact us via http://www.expdigital.co.uk
 */

#include "CClassWriterWindow.hpp"
#include "CClassWriter.hpp"
using Exponent::ClassWrite::CClassWriter;
using Exponent::ClassWrite::CClassWriterWindow;

//	===========================================================================
/*
 * Make sure that you always include this at the top of your source file. It makes sure that your class has its type info correctly initialised
 */
EXPONENT_CLASS_IMPLEMENTATION(CClassWriterWindow, CWindow);

//	===========================================================================
CClassWriterWindow::CClassWriterWindow(const CRect &size) : CWindow(size)
{
	/*
	 * This macro increments the class creation count for the class.
	 * Each counted object should call this in any and all constructors.
	 * It should be noted that this doesnt have anything to do with the reference count of the object,
	 * which is stored seperately. If you are going to reference this class in future, you must call referenced on it
	 */
	EXPONENT_CLASS_CONSTRUCTION(CClassWriterWindow);

	this->createControls();
	this->layoutRadioButtons();
	this->setupDrawingAttributes();
	this->addControlsToRoot();

	/*
	 * This next line isnt really necessary, but i put it in, cos i really like the monaco font on mac.
	 * Its not available directly on windows, which is a shame :(
	 */
#ifndef WIN32
	CFont::CFONT_SYSTEM_FONT->setFont("Monaco", 10, false, false, false);
#endif
}

//	===========================================================================
CClassWriterWindow::~CClassWriterWindow()
{
	/*
	 * This macro decrements the class creation count for the class.
	 * Each counted object should call this in its destructor
	 * It should be noted that this doesnt have anything to do with the reference count of the object,
	 * which is stored seperately. If you are finished with a class call dereferenced on it
	 */
	EXPONENT_CLASS_DESTRUCTION(CClassWriterWindow);
}

//	===========================================================================
void CClassWriterWindow::createControls()
{
	/*
	 * You will notice that we are using automatic pointers here like a normal pointer, but that the pointers that are created are never deleted
	 * directly. This is because when the automatic pointers are deleted automatically (taking their pointers with them) when the class is destroyed
	 */
	m_outputFolder	   = new CTextEdit(		  this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(  5,  5, 350,  20), "Enter the output folder here",			  NULL);
	m_name			   = new CTextEdit(		  this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(  5, 30, 400,  20), "Enter the object name here",			  NULL);
	m_namespaceEditBox = new CTextEdit(		  this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(  5, 55, 400,  20), "Enter the namespace here delimited by :", NULL);
	m_objectType	   = new CRadioButton(	  this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(  5, 80,  25, 100), 4, NULL);
	m_parentType	   = new CRadioButton(	  this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(150, 80,  25, 125), 5, NULL);
	m_doItButton	   = new CMomentaryButton(this->getControlRoot(), e_doItButton,						 CRect(300, 80, 105, 105), this);
	m_browseButton     = new CMomentaryButton(this->getControlRoot(), e_browseButton,					 CRect(360, 5,   45,  20), this);
}

//	===========================================================================
void CClassWriterWindow::layoutRadioButtons()
{
	// Get the button arrays
	TCountedPointerArray<IControl> *objectButtons = m_objectType->getButtons();
	TCountedPointerArray<IControl> *parentButtons = m_parentType->getButtons();

	// Store the label text
	CString types[4]   = { "Class",   "Interface",		"Struct", "Template" };
	CString parents[5] = { "Nothing", "ICountedObject", "CCountedObject", "CControl", "CControlPanel" };

	// Setup the class type buttons and labels
	for (long i = 0; i < 4; i++)
	{
		// Get the button
		IControl *button = objectButtons->elementAtIndex(i);

		// Check its valid
		if (button == NULL)
		{
			continue;
		}

		// Set the area
		button->setArea(CRect(0, i * 25, 25, 25));

		// Now add a label for that button
		/*
		 * Notice that we adding an annoymous pointer here.
		 * This is completely safe in the Infinity API, assuming that the function reference counts the pointer
		 * When a pointer is constructed its reference count is zero (0), when passed in to the function its reference count gets incremented to 1
		 * Note, there are a couple of caveats:
		 * 1. Dont ever pass the address of a local variable
		 * 2. Make sure the function is reference counting
		 * 3. You wont have direct access after this to the object.
		 */
		this->getControlRoot()->addControl(new CTextLabel(this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(32, 80 + (i * 25), 100, 25), types[i]));
	}

	// Now layout the button types
	for (long i = 0; i < 5; i++)
	{
		// Get the button
		IControl *button = parentButtons->elementAtIndex(i);

		// Check its valid
		if (button == NULL)
		{
			continue;
		}

		// Set the area
		button->setArea(CRect(0, i * 25, 25, 25));

		// Now add a label for that button
		this->getControlRoot()->addControl(new CTextLabel(this->getControlRoot(), CControl::CCONTROL_NO_ID_REQUIRED, CRect(177, 80 + (i * 25), 100, 25), parents[i]));
	}
}

//	===========================================================================
void CClassWriterWindow::addControlsToRoot()
{
	/*
	 * All controls have to be added to a control root. Here we are adding to the window root,
	 * but we could instead create a panel and add to that instead
	 */
	this->getControlRoot()->addControl(m_outputFolder.getMutablePointer());
	this->getControlRoot()->addControl(m_objectType.getMutablePointer());
	this->getControlRoot()->addControl(m_parentType.getMutablePointer());
	this->getControlRoot()->addControl(m_name.getMutablePointer());
	this->getControlRoot()->addControl(m_namespaceEditBox.getMutablePointer());
	this->getControlRoot()->addControl(m_doItButton.getMutablePointer());
	this->getControlRoot()->addControl(m_browseButton.getMutablePointer());
}

//	===========================================================================
void CClassWriterWindow::setupDrawingAttributes()
{
	m_outputFolder->doDefaultDrawing(true);
	m_name->doDefaultDrawing(true);
	m_namespaceEditBox->doDefaultDrawing(true);

	m_outputFolder->setSingleClickEditable();
	m_name->setSingleClickEditable();
	m_namespaceEditBox->setSingleClickEditable();

	m_doItButton->setLabelText("Do It!");
	m_browseButton->setLabelText("Browse");
}

//	===========================================================================
void CClassWriterWindow::handleActionEvent(const CActionEvent &event)
{
	/*
	 * An action event is genearted by controls when they change their state.
	 * Its their way of saying 'Ive changed, you might want to do something about it!'
	 */
	switch(event.getControl()->getUniqueId())
	{
		case e_browseButton:
			{
				CSystemString filename;
				if (CDialog::openFolderDialog(filename, "Please choose your output folder"))
				{
					m_outputFolder->setText(filename);
				}
			}
		break;
		case e_doItButton:
			this->doProcess();
		break;
	}
}

//	===========================================================================
void CClassWriterWindow::doProcess()
{
	// First we get the output folder and check that its a valid, existing path
	CSystemString outputFolder = m_outputFolder->getText();
	if (!CFileManager::isFolderPath(outputFolder))
	{
		CDialog::notifyUser("The output path selected is not an existing folder / is not a folder!", "ClassWrite path error", true);
		return;
	}

	// Next we get the name of the object
	CString objectName = m_name->getText();

	// Now get the namespace and chop it up
	TStringCountedPointerArray namespaces;
	this->tokeniseNamespaces(namespaces, m_namespaceEditBox->getText());

	// Select the class type and parent type
	CClassWriter::EClassType  classType  = (CClassWriter::EClassType) m_objectType->getSelectedIndex();
	CClassWriter::EParentType parentType = (CClassWriter::EParentType)m_parentType->getSelectedIndex();

	// Write the documents
	CClassWriter classWriter(objectName, classType, parentType, namespaces, outputFolder);

	// Notify the user that everything is finished and complete!
	CDialog::notifyUser("Process is complete", "ClassWrite process complete");
}

//	===========================================================================
void CClassWriterWindow::tokeniseNamespaces(TStringCountedPointerArray &namespaces, const CString &arguments)
{
	CStringTokeniser tokeniser(arguments, ':');
	if (tokeniser.getNumberOfTokens() <= 1)
	{
		namespaces.addElement(new CString(arguments));
	}
	else
	{
		while(tokeniser.hasMoreTokens())
		{
			CString *string = new CString;
			if (tokeniser.getNextToken(*string))
			{
				namespaces.addElement(string);
			}
			else
			{
				FORGET_COUNTED_OBJECT(string);
			}
		}
	}
	namespaces.reorder();
}